#include "Ticketing.h"
#include "SignInSony.h"
#include "MessagePipe.h"
#include "ErrorCodesSony.h"

using namespace sce::Toolkit::NP;
using namespace sce::Toolkit::NP::Utilities;

namespace UnityPlugin
{
#if NP_HAS_TICKETING
	Ticketing gTicketing;

	DO_EXPORT( bool, PrxTicketingIsBusy ) ()
	{
		return gTicketing.IsBusy();
	}

	DO_EXPORT( bool, PrxTicketingGetLastError) (ResultCode* result)
	{
		return gTicketing.GetLastError(result);
	}

	DO_EXPORT( ErrorCode, PrxTicketingRequestTicket) ()
	{
		return gTicketing.RequestTicket();
	}

	DO_EXPORT( ErrorCode, PrxTicketingRequestCachedTicket) ()
	{
		return NP_ERR_NOT_SUPPORTED;
	}

	DO_EXPORT( ErrorCode, PrxTicketingGetTicket) (Ticket* ticket)
	{
		return gTicketing.GetTicket(ticket);
	}

	DO_EXPORT( ErrorCode, PrxTicketingGetTicketInfo) (const Ticket* ticket, TicketInfo* info)
	{
		return gTicketing.GetTicketInfo(ticket, info);
	}

	DO_EXPORT( ErrorCode, PrxTicketingGetEntitlementList) (const Ticket* ticket, TicketEntitlementArray* result)
	{
		return gTicketing.GetTicketEntitlementList(ticket, result);
	}


	Ticketing::Ticketing()
		: m_Busy(false)
		, m_LastResult("Ticketing")
	{
		memset(&m_Ticket, 0, sizeof(Ticket));
	}

	Ticketing::~Ticketing()
	{
	}

	bool Ticketing::ProcessEvent(const sce::Toolkit::NP::Event& event)
	{
		SimpleLock::AutoLock lock(m_Lock);
		bool handled = false;

		switch(event.event)
		{
			case Event::authGotTicket:
				// On PS3 this is a bit rubbish; the first time we get authGotTicket there is no result, the next
				// time there is. The PS3 SCE example gets around this by not using this event, instead it just sits in a loop
				// until hasResult() returns true. Fortunatly it seems we get two authGotTicket events, one with no result, then
				// another which does have the result so all we need to do is ignore the one where there is no result.
				if(m_FutureTicket.hasResult())
				{
					m_Ticket.dataSize = m_FutureTicket.get()->size;
					free(m_Ticket.data);
					m_Ticket.data = (UInt8*)malloc(m_Ticket.dataSize);
					memcpy(m_Ticket.data, m_FutureTicket.get()->buffer, m_Ticket.dataSize);
					Messages::AddMessage(Messages::kNPToolKit_TicketingGotTicket);
					m_Busy = false;
				}
				handled = true;
				break;
				

			case Event::authError:
				m_LastResult.SetResultSCE(event.returnCode, true, __FUNCTION__, __LINE__);
				Messages::AddMessage(Messages::kNPToolKit_TicketingError);
				m_Busy = false;
				handled = true;
				break;

			default:
				UnityPlugin::Messages::LogWarning("Auth event not handled: event=%d\n", event.event);
				break;
		}

		return handled;
	}

	bool Ticketing::IsBusy()
	{
		SimpleLock::AutoLock lock(m_Lock);
		return m_Busy;
	}
	
	ErrorCode Ticketing::RequestTicket()
	{
		if(IsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();

		m_FutureTicket.reset();
		int ret = Auth::Interface::getTicket(&m_FutureTicket);
		if (ret < 0)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}	
		
		m_Busy = true;
		return m_LastResult.GetResult();
	}


	ErrorCode Ticketing::GetTicket(Ticket* ticket)
	{
		if(IsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();

		*ticket = m_Ticket;

		return m_LastResult.GetResult();
	}

	UInt64 npTime2RtcTick(SceNpTime t)
	{
		// Convert npTime to .NET DateTime compatible ticks.
		const UInt64 baseTicks = 621355968000000000ULL;
		t *= 10000;
		UInt64 rTicks = t + baseTicks;
		return rTicks;
	}

	int Ticketing::NpAuthGetTicketParam(const Ticket* /*ticket*/, int paramID, SceNpTicketParam* param)
	{
		return sceNpManagerGetTicketParam(paramID, param);
	}

	int Ticketing::NpAuthGetEntitlementIdList(const Ticket* /*ticket*/, SceNpEntitlementId* entIds, int count)
	{
		return sceNpManagerGetEntitlementIdList(entIds, count);
	}

	int Ticketing::NpAuthGetEntitlementById(const Ticket* /*ticket*/, const SceNpEntitlementId& entId, SceNpEntitlement* ent)
	{
		return sceNpManagerGetEntitlementById((const char*)entId.data, ent);
	}

	ErrorCode Ticketing::GetTicketEntitlementList(const Ticket* ticket, TicketEntitlementArray* result)
	{
		if(IsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}
		SimpleLock::AutoLock lock(m_Lock);

		m_LastResult.Reset();

		// Get the number of entitlements in the ticket.
		int count = NpAuthGetEntitlementIdList(ticket, NULL, 0);
		if(count < 0)
		{
			return m_LastResult.SetResultSCE(count, true, __FUNCTION__, __LINE__);
		}

		m_Entitlements.clear();

		if(count > 0)
		{
			// Get the entitlement IDs from the ticket.
			SceNpEntitlementId entIds[count];
			int ret = NpAuthGetEntitlementIdList(ticket, entIds, count);
			if(ret < 0)
			{
				return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
			}

			// Get the number of entitlements from the ticket.
			for(int i = 0; i < count; i++)
			{
				SceNpEntitlement ent;
				ret = NpAuthGetEntitlementById(ticket, entIds[i], &ent);
				if(ret < 0)
				{
					return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
				}
				m_Entitlements.push_back(ent);
			}
		}

		m_EntitlementsResult.clear();
		for(int i=0; i<m_Entitlements.size(); i++)
		{
			TicketEntitlement entitlement;
			SceNpEntitlement& info = m_Entitlements[i];
			entitlement.id = (const char*)&info.id.data[0];
			entitlement.createdDate = info.created_date;
			entitlement.expireDate = info.expire_date;
			entitlement.type = info.type;
			entitlement.remainingCount = info.remaining_count;
			entitlement.consumedCount = info.consumed_count;
			m_EntitlementsResult.push_back(entitlement);
		}
		result->count = m_EntitlementsResult.size();
		result->data = (result->count > 0) ? &m_EntitlementsResult[0] : NULL;

		return m_LastResult.GetResult();
	}

	ErrorCode Ticketing::GetTicketInfo(const Ticket* ticket, TicketInfo* info)
	{
		SceNpTicketParam p;

		m_LastResult.Reset();

		// Serial ID
		int ret = NpAuthGetTicketParam(ticket, SCE_NP_TICKET_PARAM_SERIAL_ID, &p);
		if(ret < 0)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}
		memcpy(m_InfoData.serialID, p.data, SCE_NP_TICKET_SERIAL_ID_SIZE);
		info->serialIDSize = SCE_NP_TICKET_SERIAL_ID_SIZE;
		info->serialID = m_InfoData.serialID;

		// Issuer ID
		ret = NpAuthGetTicketParam(ticket, SCE_NP_TICKET_PARAM_ISSUER_ID, &p);
		if(ret < 0)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}
		info->issuerID = p.u32;

		// Issued Date
		ret = NpAuthGetTicketParam(ticket, SCE_NP_TICKET_PARAM_ISSUED_DATE, &p);
		if(ret < 0)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}
		// This represents time in the POSIX time_t format, as an accumulation of milliseconds from January 1, 1970 (GMT).
		info->issuedDate = npTime2RtcTick(p.u64);

		// Expire Date
		ret = NpAuthGetTicketParam(ticket, SCE_NP_TICKET_PARAM_EXPIRE_DATE, &p);
		if(ret < 0)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}
		info->expireDate = npTime2RtcTick(p.u64);

		// Account ID
		ret = NpAuthGetTicketParam(ticket, SCE_NP_TICKET_PARAM_SUBJECT_ACCOUNT_ID, &p);
		if(ret < 0)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}
		info->subjectAccountID = p.u64;

		// Online ID
		ret = NpAuthGetTicketParam(ticket, SCE_NP_TICKET_PARAM_SUBJECT_ONLINE_ID, &p);
		if(ret < 0)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}
		SCE_NP_SUBJECT_ONLINE_ID_GET_ONLINE_ID(p.data, &m_InfoData.onlineID);
		info->subjectOnlineID = m_InfoData.onlineID.data;

		// Region
		ret = NpAuthGetTicketParam(ticket, SCE_NP_TICKET_PARAM_SUBJECT_REGION, &p);
		if(ret < 0)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}
		SCE_NP_SUBJECT_REGION_GET_COUNTRY_CODE(p.data, &m_InfoData.countryCode);
		info->countryCode = m_InfoData.countryCode.data;
		info->languageCode = SCE_NP_SUBJECT_REGION_GET_LANGUAGE_CODE(p.data);

		// Domain
		ret = NpAuthGetTicketParam(ticket, SCE_NP_TICKET_PARAM_SUBJECT_DOMAIN, &p);
		if(ret < 0)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}
		memset(m_InfoData.domain, 0, sizeof(m_InfoData.domain));
		strncpy(m_InfoData.domain, (const char*)p.data, sizeof(m_InfoData.domain));
		info->subjectDomain = m_InfoData.domain;

		// Service ID
		ret = NpAuthGetTicketParam(ticket, SCE_NP_TICKET_PARAM_SERVICE_ID, &p);
		if(ret < 0)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}
		memset(m_InfoData.serviceID, 0, sizeof(m_InfoData.serviceID));
		strncpy(m_InfoData.serviceID, (const char*)p.data, sizeof(m_InfoData.serviceID));
		info->serviceID = m_InfoData.serviceID;

		// Subject Status
		ret = NpAuthGetTicketParam(ticket, SCE_NP_TICKET_PARAM_SUBJECT_STATUS, &p);
		if(ret < 0)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}
		info->subjectAge = SCE_NP_SUBJECT_STATUS_GET_AGE(p.u32);
		info->chatDisabled = SCE_NP_SUBJECT_STATUS_IS_CHAT_DISABLED(p.u32);
		info->contentRating = SCE_NP_SUBJECT_STATUS_CONTENT_RATING(p.u32);

		// Subject Status (optional)
		ret = NpAuthGetTicketParam(ticket, SCE_NP_TICKET_PARAM_STATUS_DURATION, &p);
		if(ret < 0 && ret != SCE_NP_UTIL_ERROR_PARSER_FAILED)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}
		info->statusDuration = ret != (ret == SCE_NP_UTIL_ERROR_PARSER_FAILED) ? 0 : p.u64;

		return m_LastResult.GetResult();
	}

#else // Not supported

DO_EXPORT( bool, PrxTicketingIsBusy ) ()
{
	return false;
}

DO_EXPORT( bool, PrxTicketingGetLastError) (ResultCode* result)
{
	return false;
}

DO_EXPORT( ErrorCode, PrxTicketingRequestTicket) ()
{
	return NP_ERR_NOT_SUPPORTED;
}

DO_EXPORT( ErrorCode, PrxTicketingRequestCachedTicket) ()
{
	return NP_ERR_NOT_SUPPORTED;
}

DO_EXPORT( ErrorCode, PrxTicketingGetTicket) (Ticket* ticket)
{
	return NP_ERR_NOT_SUPPORTED;
}

DO_EXPORT( ErrorCode, PrxTicketingGetTicketInfo) (const Ticket* ticket, TicketInfo* info)
{
	return NP_ERR_NOT_SUPPORTED;
}

DO_EXPORT( ErrorCode, PrxTicketingGetEntitlementList) (const Ticket* ticket, TicketEntitlementArray* result)
{
	return NP_ERR_NOT_SUPPORTED;
}

#endif

}
